// This program is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License version 2 as
// published by the Free Software Foundation.

// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.

#ifndef DEGUB_H
#define DEGUB_H

#include "ini.h"
#include "defineCommons.h"

#define _WIN32_WINNT 0x0500
#include <windows.h>

#include <stdio.h>
#include <stdexcept>
#include <crtdbg.h>

extern FILE *df;
extern int g_con_total, g_con_num, g_arr_num;
extern int g_con_maximum, g_con_num_max, g_arr_num_max;
extern HWND g_hWnd;
extern const char *g_buildtime;

#define ZERO_OBJECT(object) ZeroMemory(&object, sizeof(object))
#define ZERO_ARRAY(array) ZeroMemory(array, sizeof(array))

#define DEGUB if(g::degub_on) degub_degub
void degub_degub(const char *format, ...);

//error handling
#define IN_FILE_ON_LINE DEGUB("In file %s on line %i\n", __FILE__, __LINE__)

//Return with Degub
#define RD(hr) {\
	DEGUB("Returning in file %s on line %i: 0x%x\n", __FILE__, __LINE__, hr); DumpGLE();return hr; }

//Win32 error handling
#define DO_GLE { IN_FILE_ON_LINE; DoGLE(); }
#define GLECUSTOM(a, b) { if((a) == (b)) DO_GLE; }
#define GLE(a) GLECUSTOM(a, 0)
#define TGLECUSTOM(a, b) { if((a) == (b)) { DumpGLE(); RD(0); } }
#define TGLE(a) TGLECUSTOM(0, a)
#define R0GLE(a) { BOOL temp = (BOOL)(a); GLE(temp); if(temp == 0) return 0; }
#define SET_AND_SHOW_LE(error) { SetLastError(error); DO_GLE; }
#define HWGLECUSTOM(a, b) if((a) == (b)) { IN_FILE_ON_LINE; ThrowGLE(); }
#define HWGLE(a) HWGLECUSTOM(a, 0)
//Reverse
#define TGLER(a) TGLERCUSTOM(0, a)
#define TGLERCUSTOM(a, b) { if((a) != (b)) { DumpGLE(); RD(0); } }
#define GLERCUSTOM(a, b) { if((a) != (b)) { DO_GLE; } }
#define GLER(a) GLERCUSTOM(a, 0)
#define HWGLERCUSTOM(a, b) if((a) != (b)) { IN_FILE_ON_LINE; ThrowGLE(); }

//DirectX error handling
#define THR(func) { HRESULT _h; if(FAILED(_h=(func))) RD(_h); }
#define HR(func) { HRESULT _h; if(FAILED(_h=(func))) { IN_FILE_ON_LINE; DoDXErr(_h); } }
#define R0HR(func) { HRESULT __hr = (func); HR(__hr) if(FAILED(__hr)) return 0; }
#define R1HR(func) { HRESULT __hr = (func); HR(__hr) if(FAILED(__hr)) return 1; }
#define SAFE_RELEASE(p) { if(p) { (p)->Release(); (p)=NULL; } }
#define INFOL_DX(hr) IN_FILE_ON_LINE; DEGUB("DirectX Error: %s\n", DXGetErrorString9(hr));
#define CHR(func) { HRESULT _hr; if(FAILED(_hr=(func))) { INFOL_DX(_hr) } }
#define HWHR(func) { HRESULT _hr; if(FAILED(_hr=(func))) { INFOL_DX(_hr)\
	throw hardware_fatal_exception(DXGetErrorString9(_hr)); } }
#define HR2GLE(gle, func) { HRESULT _hr; if(FAILED(_hr=(func))) { INFOL_DX(_hr)\
	SetLastError(gle); return false; } }
#define HWHR2GLE(gle, func) { HRESULT _hr; if(FAILED(_hr=(func))) { INFOL_DX(_hr)\
	SetLastError(gle); ThrowGLE(); } }

//Otha stuff
#define CASEDEGUB(a) case a: DEGUB(#a); break;
#define DEGUB_CASE_BREAK(con) case con: DEGUB("%s", #con); break;
#define DEGUB_CASE_NOBREAK(con) case con: DEGUB("%s", #con);
#define DUMPINT(i) DEGUB("%s=%i\n", #i, i);
#define DUMPBYTE(i) DEGUB("%s=0x%02X\n", #i, i);
#define DUMPWORD(i) DEGUB("%s=0x%04X\n", #i, i);
#define DUMPDWORD(i) DEGUB("%s=0x%08X\n", #i, i);
#define DUMPQWORD(i) DEGUB("%s=0x%016I64X\n", #i, i);
#define SAFE_DELETE(p) { if(p) { delete (p); (p)=NULL; } }
#define TFAILCUSTOM(code, func) { DWORD res=(func); if(res!=code) FAIL(res) }

#define FORCED_CRASH (*(BYTE*)0) = 0
#ifdef PUBLIC_RELEASE
#define NONPUBLIC_FORCED_CRASH
#else
#define NONPUBLIC_FORCED_CRASH FORCED_CRASH
#endif

//User errors
#define FAIL(code) { SetLastError(code); DumpGLE(); RD(0); }
#define USER_ERROR(number) (number | 0x20000000)
#define IS_USER_ERROR(code) ((code & 0x20000000) ? true : false)

//It should be enough to add an error code here only
//And soon it will be >:D
//Now it is! :)
#define UE_ERROR_CODES(macro) \
	macro(UE_GENERIC_ERROR, "Something bad's happened. Check degub.txt for more info.")\
	macro(UE_NO_INPUT, "There is no input.")\
	macro(UE_UNKNOWN_OPERATION, "Unknown operation.")\
	macro(UE_INI_WRONG_TYPE, "Wrong INIvariable type.")\
	macro(UE_INI_NOT_INITIALIZED, "INI system not initialized.")\
	macro(UE_INI_CANNOT_OPEN_FILE, "Cannot open INI file.")\
	macro(UE_INI_VAR_NOT_FOUND, "INI variable not found.")\
	macro(UE_REC_INTERNAL_ERROR, "Internal recompiler error.")\
	macro(UE_BBA_ERROR, "Internal BBA error.")\
	macro(UE_FIREWALL, "BBA Send failed. This is probably caused by a software firewall. "\
	"If you have one, deactivate it and try again.")\
	macro(UE_REC_INVALID_IA, "Invalid Instruction Address.")\
	macro(UE_THREAD_TIMEOUT, "Timeout waiting for the Thread.")\
	macro(UE_REC_ABORT, "Recompilation aborted.")\
	macro(UE_INSUFFICIENT_D3D_CAPS, "Your graphics card (or driver) does not support "\
	"operations required by WhineCube.")\
	macro(UE_WSA_ERROR, "Error using Windows Sockets.")\
	macro(UE_INVALID_DOL, "Not a valid DOL file.")\
	macro(UE_INVALID_ELF, "Not a valid ELF file.")\
	macro(UE_INCOMPATIBLE_ELF, "Not a WhineCube-compatible ELF file.")\
	macro(UE_INVALID_FILE_TYPE, "Invalid file type.")\
	macro(UE_CANNOT_OPEN_FILE, "Cannot open file.")\
	macro(UE_CANNOT_READ_FILE, "Cannot read file.")\
	macro(UE_CANNOT_WRITE_FILE, "Cannot write file.")\
	macro(UE_EARLY_EOF, "Unexpected End Of File.")\
	macro(UE_SYMBOL_NOT_FOUND, "Symbol not found.")\
	macro(UE_INVALID_FILE_SIZE, "Invalid file size.")\
	macro(UE_BAD_HRCF, "Internal error: Bad HRCF.")\
	macro(UE_DIRECT3D, "Error initializing Direct3D. "\
	"(Re)installing DirectX 9.0c might help.")\
	macro(UE_D3DDEVICE, "Error creating D3D Device. You may need a better 3D card or "\
	"newer drivers in order to run WhineCube.")\
	macro(UE_DIRECTINPUT, "Error initializing DirectInput.")\
	macro(UE_GPTHREAD_TIMEOUT, "Timeout waiting for the GP Thread.")\
	macro(UE_SHADER_COMPILATION_FAILED, "Internal error: Shader compilation failed.")\
	macro(UE_TEST, "Testink testink")
//macro(UE_DOL_LOAD_ERROR, "Error loading DOL file.")\
//macro(UE_ELF_LOAD_ERROR, "Error loading ELF file.")\
//Beware: an \ on the last line makes e3.cpp uncompilable.

#define ENUM_ERROR_CODES(code, desc) code,

enum UE_Codes { UE_NO_ERROR = USER_ERROR(0), UE_ERROR_CODES(ENUM_ERROR_CODES)
_UE_SIZE };

class Degub {
public:
	Degub();
	bool rename(const char *newname);
	bool restart(const char *newname);
	void turn_off();
	~Degub();
private:
	std::string name;
};

extern Degub g_degub;

//Exceptions
#define DECLARE_EXCEPTION(id) \
class id : public std::logic_error {\
public: id(const std::string& message) : logic_error(message) {} };

#define LAZER_EXCEPTIONS(macro) macro(rec_fatal_exception)\
	macro(interp_fatal_exception) macro(hardware_fatal_exception) macro(bouehr_exception)\
	macro(ui_exception) macro(page_fault_exception) macro(generic_fatal_exception)\
	macro(rec_abort_exception)
LAZER_EXCEPTIONS(DECLARE_EXCEPTION);

class stop_exception_base {
public:
	virtual const char* what() const = 0;
};

#define DECLARE_STOP_EXCEPTION(id) \
class id : public std::logic_error, public stop_exception_base {\
public: id(const std::string& message) : logic_error(message) {}\
	const char* what() const { return logic_error::what(); } };

#define LAZER_STOP_EXCEPTIONS(m) m(disasm_exception) m(rec_stop_exception)\
	m(emulation_stop_exception)
LAZER_STOP_EXCEPTIONS(DECLARE_STOP_EXCEPTION);

class bad_form_exception : public stop_exception_base {
public:
	const enum Type { Invalid, Unemulated } type;
	bad_form_exception(Type aType) : type(aType) {}
	const char* what() const { return type == Invalid ?
		"Invalid instruction form" : "Unemulated instruction form"; }
};

//MYASSERT
#ifdef _DEBUG
#define MYASSERT(expr) _ASSERTE(expr)
#elif defined(RELEASE_DEBUG)
#define MYASSERT(expr) throw_on_false(expr, #expr)
inline void throw_on_false(bool b, const char *expr) {
	if(!b) {
		DEGUB("(%s) ", expr);
		IN_FILE_ON_LINE;
		throw generic_fatal_exception(expr);
	}
}
#else
#define MYASSERT(expr) __assume(expr)
#endif

//functions
void DoGLE();
void DumpGLE();
void ThrowGLE();  //throws generic_fatal_exception
void DoCDEE(DWORD cdee);
void DoDXErr(HRESULT hr);
bool _TheOneMessageBox(const std::string& message);
#define THE_ONE_MESSAGE_BOX(message) { if(!_TheOneMessageBox(message)) {\
	DumpGLE(); IN_FILE_ON_LINE; } }
inline void BFE(const string& str) {
	DEGUB("Big Fat Error: %s\n", str.c_str());
	THE_ONE_MESSAGE_BOX(str);
}

std::string TranslateWM(DWORD message);	//Window Message
std::string TranslateGLE(DWORD error);	//GetLastError
std::string TranslateCDEE(DWORD error);	//CommDlgExtendedError
std::string TranslateCN(WORD note);	//Control Notification

#endif //DEGUB_H
